function mech_unit_cell_band_structure(geom, Eval_mech)

    format long
    clear all;
    close all;

    tic;
    import com.comsol.model.*
    import com.comsol.model.util.*

    model = ModelUtil.create('Model');
    model.modelNode.create('comp1');
    geometry = model.geom.create('geom1',3);

    disp('Setting Paramters... ');
        
%%   parameters and definitions
    
    % Scale factors for CAD sweep
    geom.scale_all = 1.02;
    geom.scale_small_holes = 1.0702; % vary scale_small_holes

    % Parameters of the snowflake (some of these values are scaled later by
    % geom.mech.scale)
    geom.mech.d = 220e-9;
    geom.mech.r = 180e-9*geom.scale_all;
    geom.mech.delta_r = 0; % Shift the Dirac cone in kx direction
    geom.mech.r1 = geom.mech.r + geom.mech.delta_r;
    geom.mech.r2 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.r3 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.w = 73e-9*geom.scale_all;
    geom.mech.a = 500e-9*geom.scale_all;
    geom.mech.rad_curv = 20e-9*geom.scale_all;
    geom.mech.rad_curv2 = 12e-9*geom.scale_all;
    geom.mech.resolution = 40; % Resolution of the snowflake polygon

    % Minimum distance between optical cavity hole and the snowflake hole
    geom.bridge_width = 70e-9*geom.scale_all;

    % Parameters of the optical cavity
    geom.opt.scale = 1.077*geom.scale_all; % Do not change this scaling parameter for CAD sweeps; Change scale_structure and scale_radius
    geom.opt.delta_a = -9e-9*geom.opt.scale;
    geom.opt.a1 = 425e-9*geom.opt.scale;
    geom.opt.a2 = geom.opt.a1+geom.opt.delta_a;
    geom.opt.a3 = geom.opt.a1+2*geom.opt.delta_a;
    geom.opt.a4 = geom.opt.a1+3*geom.opt.delta_a;
    geom.opt.r = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.W = 0;
    geom.opt.h = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.b = 150e-9*geom.opt.scale*geom.scale_small_holes;

    geom.opt.nh = 20;
    geom.opt.nv = 40;
    geom.opt.defect_holes = 2*geom.opt.nv;
    geom.opt.defect_width = geom.opt.r*2/geom.scale_small_holes + 30e-9;
    geom.opt.defect_length = geom.opt.a1*100;
    geom.opt.cavity_rot = 0;
    geom.opt.cavity_shift = 4; % Cavity shift from the center of triangle
    
    opt_scale_top_gap = linspace(0.8, 1.2, 31); %Scale factor of the radius in one of the triangle
%     geom.opt.scale_top_gap = opt_scale_top_gap(25);
%     geom.opt.scale_top_gap = 1; % no mass (zero bulk band gap)
    geom.opt.scale_top_gap = 0.783333; % with mass (finite bulk band gap)

    % Determining the scale factor of the snowflake so that optical cavity fits
    % inside the structure
    n = linspace(1,20,20);
    s = ((n+1)*geom.opt.a1 + 2*geom.opt.a2 + 2*geom.opt.a3 + geom.opt.a4)*sqrt(3)/geom.mech.a;
    index = 14; % 13 bad, 14 good
    num = n(index);
    geom.mech.scale = s(index);

    % Scaling the parameters of the snowflake by the above scale factor
    geom.mech.amech = geom.mech.a*geom.mech.scale;
    geom.mech.r = geom.mech.r*geom.mech.scale;
    geom.mech.r1 = geom.mech.r1*geom.mech.scale;
    geom.mech.r2 = geom.mech.r2*geom.mech.scale;
    geom.mech.r3 = geom.mech.r3*geom.mech.scale;
    geom.mech.w = geom.mech.w*geom.mech.scale;
    geom.mech.a = geom.mech.a*geom.mech.scale;
    geom.mech.rad_curv = geom.mech.rad_curv*geom.mech.scale;
    geom.mech.rad_curv2 = geom.mech.rad_curv2*geom.mech.scale;

    % Height of the snowflake triangle (not the triangle where topological edge
    % state is travelling)
    geom.mech.gap_defect_length = geom.mech.amech*sqrt(3)/2 - geom.mech.w*3/2;

    % Mesh parameters
    Eval_mech.hmax = 90e-9;
    Eval_mech.hmin = 25e-9;
    
    % Mirror symmetry about xy plane
    Eval_mech.symm_z = 1;
    
    % .mph and .mat output files
    Eval_mech.filename = 'exp_comp_bulk_with_mass';
    
    % Path along the k-space
    Eval_mech.res = 25; % Resolution of the path
    Eval_mech.fine_res = 5; % Resolution of the path near K-point
    G = [0,0];
    M = pi/geom.mech.amech*[1, 1/sqrt(3)];
    K = 4*pi/geom.mech.amech*[1/3, 0];
    
    kx1 = linspace(G(1),M(1),Eval_mech.res);
    kx2 = linspace(M(1),K(1),Eval_mech.res);
    kx3 = linspace(kx2(end-1),K(1),Eval_mech.fine_res);
    kx5 = linspace(K(1),G(1),Eval_mech.res);
    kx4 = linspace(K(1),kx5(2),Eval_mech.fine_res);

    ky1 = linspace(G(2),M(2),Eval_mech.res);
    ky2 = linspace(M(2),K(2),Eval_mech.res);
    ky3 = linspace(ky2(end-1),K(2),Eval_mech.fine_res);
    ky5 = linspace(K(2),G(2),Eval_mech.res);
    ky4 = linspace(K(2),ky5(2),Eval_mech.fine_res);

    Eval_mech.kx = [kx1 kx2(2:end-1) kx3(2:end) kx4(2:end) kx5(3:end)];
    Eval_mech.ky = [ky1 ky2(2:end-1) ky3(2:end) ky4(2:end) ky5(3:end)];
    
    % Comsol output parameters
    Eval_mech.NumberOfFrequencies = 45;
    Eval_mech.CentralFrequency = 0;
    
    Eval_mech.bands = zeros(Eval_mech.NumberOfFrequencies, size(Eval_mech.kx,2));
    
%%   COMSOL-GLOBAL PARAMETERS
  
    model.param.set('d', num2str(geom.mech.d));
    model.param.set('r', num2str(geom.mech.r));
    model.param.set('w', num2str(geom.mech.w));
    model.param.set('a', num2str(geom.mech.a));
    model.param.set('a1', num2str(geom.opt.a1));
    model.param.set('a2', num2str(geom.opt.a2));
    model.param.set('a3', num2str(geom.opt.a3));
    model.param.set('a4', num2str(geom.opt.a4));
    model.param.set('r', num2str(geom.opt.r));
    model.param.set('W', num2str(geom.opt.W));
    model.param.set('height', num2str(geom.opt.h));
    model.param.set('b', num2str(geom.opt.b));
    model.param.set('hmax', num2str(Eval_mech.hmax));
    model.param.set('hmin', num2str(Eval_mech.hmin));

    mphsave(model, Eval_mech.filename);

%%   COMSOL-GEOMETRY
    disp('Building Geometry... ');
    
    workplane = geometry.feature.create('wp1','WorkPlane');
    
    % Hexagon of the small unit cell
    hex_x = [0 0 geom.mech.amech/2 geom.mech.amech geom.mech.amech geom.mech.amech/2 0];
    hex_y = [-geom.mech.amech/(2*sqrt(3)) geom.mech.amech/(2*sqrt(3)) geom.mech.amech/sqrt(3) geom.mech.amech/(2*sqrt(3)) -geom.mech.amech/(2*sqrt(3)) -geom.mech.amech/sqrt(3) -geom.mech.amech/(2*sqrt(3))];
    
    hex2_x = hex_x;
    hex2_y = hex_y + geom.mech.amech/sqrt(3);

    % Rectangular area in the unit cell where the geometry has to be specified
    rect1_x = [geom.mech.amech/4 geom.mech.amech/4 geom.mech.amech*3/4 geom.mech.amech*3/4 geom.mech.amech/4];
    rect1_y = [0 geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 0 0];

    rect2_x = [geom.mech.amech*3/4 geom.mech.amech*3/4 geom.mech.amech geom.mech.amech geom.mech.amech*3/4];
    rect2_y = [-geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 -geom.mech.amech*sqrt(3)/2 -geom.mech.amech*sqrt(3)/2];
    
    % Draw snowflakes
    % angle = -60;
    x_cen = geom.mech.amech/2;
    y_cen = 0;
    [snowflake_cen_x, snowflake_cen_y] = coordinates_snowflake(x_cen, y_cen, geom.mech.r, geom.mech.w);

    mech = workplane.geom.create('pol2', 'Polygon');
    mech.set('source', 'table');
    for pol_i=1:length(snowflake_cen_x(1:end-1))
        mech.setIndex('table', num2str(snowflake_cen_x(pol_i),'%10.9e'), pol_i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(snowflake_cen_y(pol_i),'%10.9e'), pol_i-1, 1);
    end
    
    % Fillet the snowflake corners
    epsilon = 100e-9;
    disksel = workplane.geom.create('disksel1', 'DiskSelection');
    disksel.set('entitydim', 0);
    disksel.set('posx', num2str(geom.mech.amech/2,'%10.9e'));
    disksel.set('posy', '0');
    disksel.set('r', num2str(geom.mech.w + epsilon,'%10.9e'));
    disksel.set('condition', 'inside');
    
    disksel = workplane.geom.create('disksel2', 'DiskSelection');
    disksel.set('entitydim', 0);
    disksel.set('posx', num2str(geom.mech.amech/2,'%10.9e'));
    disksel.set('posy', '0');
    disksel.set('r', num2str(sqrt(geom.mech.r^2 + (geom.mech.w/2)^2) + epsilon,'%10.9e'));
    disksel.set('rin', num2str(geom.mech.w + epsilon,'%10.9e'));
    disksel.set('condition', 'inside');
    
    fil = workplane.geom.create('fil1', 'Fillet');
    fil.set('radius', num2str(geom.mech.rad_curv2,'%10.9e'));
    fil.selection('point').named('disksel1');
    
    fil = workplane.geom.create('fil2', 'Fillet');
    fil.set('radius', num2str(geom.mech.rad_curv,'%10.9e'));
    fil.selection('point').named('disksel2');
    
    [snowflake_cen_x, snowflake_cen_y] = coordinates_round_snowflake_comsol(x_cen, y_cen, geom.mech.r, geom.mech.w, geom.mech.rad_curv, geom.mech.rad_curv2, geom.mech.resolution);
    
    snowflake_x4 = snowflake_cen_x + geom.mech.amech/2;
    snowflake_y4 = snowflake_cen_y + geom.mech.amech*sqrt(3)/2;
    
    % Position of gap defect
%     x_cen = geom.mech.amech/2;
%     y_cen = -geom.mech.amech/sqrt(3);
%     geom.mech.gap_defect_x = [x_cen-geom.mech.gap_defect_width/2 x_cen-geom.mech.gap_defect_width/2 x_cen+geom.mech.gap_defect_width/2 x_cen+geom.mech.gap_defect_width/2];
%     geom.mech.gap_defect_y = [y_cen-geom.mech.gap_defect_length/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen-geom.mech.gap_defect_length/3];
    
    % Coordinates of triangle where the hole sizes are scaled in order to create topological bandgap
    x_cen = geom.mech.amech;
    y_cen = geom.mech.amech/(2*sqrt(3));
    geom.mech.tri_defect_x = [x_cen-geom.mech.gap_defect_length/sqrt(3) x_cen x_cen+geom.mech.gap_defect_length/sqrt(3)];
    geom.mech.tri_defect_y = [y_cen-geom.mech.gap_defect_length/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen-geom.mech.gap_defect_length/3];
    
    % Position of optical cavity
    x_cen = geom.mech.amech/2;
    y_cen = geom.mech.amech/sqrt(3) + geom.opt.cavity_shift*geom.opt.a1/2;
    temp_x = [x_cen-geom.opt.defect_width/2 x_cen-geom.opt.defect_width/2 x_cen+geom.opt.defect_width/2 x_cen+geom.opt.defect_width/2];
    temp_y = [y_cen-geom.opt.defect_length/2 y_cen+geom.opt.defect_length/2 y_cen+geom.opt.defect_length/2 y_cen-geom.opt.defect_length/2];
    [geom.opt.defect_x, geom.opt.defect_y] = rotate(temp_x, temp_y, x_cen, y_cen, geom.opt.cavity_rot);

    % plot(geom.opt.defect_x, geom.opt.defect_y, 'color', 'red', 'LineWidth', 0.5)

    delta2 = [geom.opt.a1*ones(1,geom.opt.nv-5), ...
        (geom.opt.a1+geom.opt.a2)/2, ... %1
        geom.opt.a2, ... %2
        (geom.opt.a2+geom.opt.a3)/2, ... %3
        geom.opt.a3, ... %4
        (geom.opt.a3+geom.opt.a4)/2, ... %5
        geom.opt.a4, ... %6
        (geom.opt.a3+geom.opt.a4)/2, ... %7
        geom.opt.a3, ... %8
        (geom.opt.a2+geom.opt.a3)/2, ... %9
        geom.opt.a2, ... %10
        (geom.opt.a1+geom.opt.a2)/2, ... %11
        geom.opt.a1*ones(1,geom.opt.nv-5)];

    delta1 = [geom.opt.a1*ones(1,geom.opt.nv-6), ...
        (geom.opt.a1+(geom.opt.a1+geom.opt.a2)/2)/2, ... %1
        (geom.opt.a2+(geom.opt.a1+geom.opt.a2)/2)/2, ... %2
        (geom.opt.a2+(geom.opt.a2+geom.opt.a3)/2)/2, ... %3
        (geom.opt.a3+(geom.opt.a2+geom.opt.a3)/2)/2, ... %4
        (geom.opt.a3+(geom.opt.a3+geom.opt.a4)/2)/2, ... %5
        (geom.opt.a4+(geom.opt.a3+geom.opt.a4)/2)/2, ... %6
        (geom.opt.a4+(geom.opt.a3+geom.opt.a4)/2)/2, ... %7
        (geom.opt.a3+(geom.opt.a3+geom.opt.a4)/2)/2, ... %8
        (geom.opt.a3+(geom.opt.a2+geom.opt.a3)/2)/2, ... %9
        (geom.opt.a2+(geom.opt.a2+geom.opt.a3)/2)/2, ... %10
        (geom.opt.a2+(geom.opt.a1+geom.opt.a2)/2)/2, ... %11
        (geom.opt.a1+(geom.opt.a1+geom.opt.a2)/2)/2, ... %12
        geom.opt.a1*ones(1,geom.opt.nv-6)];
    x = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    y = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    rad = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    radb = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);

    for j = -geom.opt.nh:geom.opt.nh
        if j==-geom.opt.nh
            if rem(geom.opt.nv,2) == 0
                x(:,j+geom.opt.nh+1) = -geom.opt.nh*geom.opt.a1*sqrt(3)/2 - geom.opt.W/2;
            else
                x(:,j+geom.opt.nh+1) = -(geom.opt.nh+1)*geom.opt.a1*sqrt(3)/2 - geom.opt.W/2;
            end
        else
            if j==0 || j==1
                x(:,j+geom.opt.nh+1) = x(:,j+geom.opt.nh) + (geom.opt.a1*sqrt(3) + geom.opt.W)/2;
            else
                x(:,j+geom.opt.nh+1) = x(:,j+geom.opt.nh) + geom.opt.a1*sqrt(3)/2;
            end
        end

        for i=-geom.opt.nv:geom.opt.nv
            rad(i+geom.opt.nv+1,j+geom.opt.nh+1) = geom.opt.r;

            if i==-geom.opt.nv
                if rem(j,2)==0
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = -sum(delta1(1:geom.opt.nv));
                else
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = -sum(delta2(1:geom.opt.nv))-geom.opt.a3/2;
                end
            else
                if rem(j,2)==0
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = y(i+geom.opt.nv,j+geom.opt.nh+1)+delta1(i+geom.opt.nv);
                else
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = y(i+geom.opt.nv,j+geom.opt.nh+1)+delta2(i+geom.opt.nv);
                end
            end
        end
    end

    temp_x = x+x_cen;
    temp_y = y+y_cen;
    [x, y] = rotate(temp_x, temp_y, x_cen, y_cen, geom.opt.cavity_rot);

    % Draw cavity holes in region 1 of geometry schematic
    count = 1;
    count1 = 1;
    for j = -geom.opt.nh:geom.opt.nh
        for i = -geom.opt.nv:geom.opt.nv
            x1 = x(i+geom.opt.nv+1,j+geom.opt.nh+1);
            y1 = y(i+geom.opt.nv+1,j+geom.opt.nh+1);

            r1 = rad(i+geom.opt.nv+1,j+geom.opt.nh+1);
            rh = geom.opt.h;
            rb = geom.opt.b;

            if inpolygon(x1, y1, geom.opt.defect_x, geom.opt.defect_y) == 0
                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi)/geom.scale_small_holes;
                yy = y1 + r1*sin(phi)/geom.scale_small_holes;
            else
                y1 = y(i+geom.opt.nv+1,j+1+geom.opt.nh+1);
                phi = linspace(0,2*pi,37);
                xx = x1 + rb*cos(phi)/geom.scale_small_holes;
                yy = y1 + rh*sin(phi)/geom.scale_small_holes;
            end

            for k=1:length(xx)-1
                check(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_cen_x,snowflake_cen_y));
            end
            
            in_scale_tri = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x, geom.mech.tri_defect_y);
            in_scale_tri2 = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x-geom.mech.amech, geom.mech.tri_defect_y);
            in_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.opt.defect_x, geom.opt.defect_y);
            in_rect1 = inpolygon(xx(1:end-1), yy(1:end-1), rect1_x, rect1_y);
            in_uc = inpolygon(xx(1:end-1), yy(1:end-1), hex_x, hex_y);
            in = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_cen_x, snowflake_cen_y);

            dist1 = x1 - geom.mech.amech/4;
            dist2 = geom.mech.amech*3/4 - x1;

            if min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count1 = count1+1;

            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)~=0 && geom.opt.h ~=0 && geom.opt.b ~=0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end
                
                inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist2<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (dist2 - geom.opt.a1*sqrt(3)/2)/2;
                
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)==length(xx)-1
                    inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                else
                    inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                end
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (geom.opt.a1*sqrt(3)/2 - dist1)/2;
                
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)==length(xx)-1
                    inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                else
                    inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                end
            end
        end
    end
    
    uni = workplane.geom.create('uni_region1', 'Union');
    uni.selection('input').set(inp);
    
    uni = workplane.geom.create('uni_region1_bdd', 'Union');
    uni.selection('input').set(inp1);
    
    copy = workplane.geom.create('cp_region1_bdd_1', 'Copy');
    copy.selection('input').set({'uni_region1_bdd'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(-geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    copy = workplane.geom.create('cp_region1_bdd_2', 'Copy');
    copy.selection('input').set({'uni_region1_bdd'});
    copy.set('displx', num2str(-geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(-geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    % Constructing cavity holes in region 2 of geometry schematic
    geom.opt.defect_x = geom.opt.defect_x + geom.mech.amech/2;
    geom.opt.defect_y = geom.opt.defect_y - geom.mech.amech*sqrt(3)/2;
    
%     geom.mech.gap_defect_x = geom.mech.gap_defect_x + geom.mech.amech/2;
%     geom.mech.gap_defect_y = geom.mech.gap_defect_y + geom.mech.amech*sqrt(3)/2;

    clear inp inp1;
    count = 1;
    count1 = 1;
    for j = -geom.opt.nh:geom.opt.nh
        for i = -geom.opt.nv:geom.opt.nv
            x1 = x(i+geom.opt.nv+1,j+geom.opt.nh+1) + geom.mech.amech/2;
            y1 = y(i+geom.opt.nv+1,j+geom.opt.nh+1) - geom.mech.amech*sqrt(3)/2;

            r1 = rad(i+geom.opt.nv+1,j+geom.opt.nh+1);
            rh = geom.opt.h;
            rb = geom.opt.b;

            if inpolygon(x1, y1, geom.opt.defect_x, geom.opt.defect_y) == 0
                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi)/geom.scale_small_holes;
                yy = y1 + r1*sin(phi)/geom.scale_small_holes;
            else
                y1 = y(i+geom.opt.nv+1,j+1+geom.opt.nh+1) - geom.mech.amech*sqrt(3)/2;
                phi = linspace(0,2*pi,37);
                xx = x1 + rb*cos(phi)/geom.scale_small_holes;
                yy = y1 + rh*sin(phi)/geom.scale_small_holes;
            end

            for k=1:length(xx)-1
                check(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_cen_x,snowflake_cen_y));
                check1(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_x4,snowflake_y4));
            end
            
            in_scale_tri = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x, geom.mech.tri_defect_y);
            in_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.opt.defect_x, geom.opt.defect_y);
%             in_gap_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.gap_defect_x, geom.mech.gap_defect_y);
            in_rect2 = inpolygon(xx(1:end-1), yy(1:end-1), rect2_x, rect2_y);
            in_uc = inpolygon(xx(1:end-1), yy(1:end-1), hex_x, hex_y);
            in = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_cen_x, snowflake_cen_y);
            in1 = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_x4, snowflake_y4);

            dist = x1 - geom.mech.amech*3/4;

%             if min(check)>geom.bridge_width && sum(in_gap_defect)~=0 && sum(in1)==0
%                 x1 = x1 - geom.mech.amech/2;
%                 y1 = y1 - geom.mech.amech*sqrt(3)/2;
%                 
%                 workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
%                 workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
%                 workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
% 
%                 inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
%                 count1 = count1+1;
            
            if min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect2)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;

            elseif min(check)>geom.bridge_width && sum(in_rect2)>0 && sum(in)==0 && sum(in_uc)>0 && sum(in_defect)~=0 && geom.opt.h ~=0 && geom.opt.b ~=0 && dist>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;

            elseif min(check)>geom.bridge_width && min(xx)<geom.mech.amech*3/4 && sum(in_uc)>0 && max(xx)>geom.mech.amech*3/4 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)==0
                x1 = geom.mech.amech*3/4;
                
                if sum(in_uc)~=length(xx)-1
                    y1 = geom.mech.amech/sqrt(3) - abs((x1 - geom.mech.amech/2))/sqrt(3);
                    
                    x1 = x1 - geom.mech.amech/2;
                    y1 = y1 - geom.mech.amech*sqrt(3)/2;
                end

                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi);
                yy = y1 + r1*sin(phi);

                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)~=length(xx)-1
                    inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                else
                    inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                end
                
            elseif min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect2)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (geom.opt.a1*sqrt(3)/2 - dist)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check1)>geom.bridge_width && sum(in_uc)==0 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)==0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                            
            elseif min(check1)>geom.bridge_width && sum(in_uc)==0 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)~=0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect2)>0 && sum(in)==0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count1 = count1+1;
            end
        end
    end
    
    uni = workplane.geom.create('uni_region2_bdd', 'Union');
    uni.selection('input').set(inp1);
    
    copy = workplane.geom.create('cp_region2_bdd', 'Copy');
    copy.selection('input').set({'uni_region2_bdd'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    uni = workplane.geom.create('uni_region2', 'Union');
    uni.selection('input').set(inp);
    
    mir = workplane.geom.create('mir1', 'Mirror');
    mir.set('keep', true);
    mir.selection('input').set({'uni_region2' 'uni_region2_bdd' 'cp_region2_bdd'});
    mir.set('pos', {num2str(geom.mech.amech/2,'%10.9e') '0'});
    
    uni = workplane.geom.create('uni1', 'Union');
    uni.selection('input').set({'fil2' 'uni_region1' 'uni_region1_bdd' 'cp_region1_bdd_1' 'cp_region1_bdd_2' 'uni_region2' 'uni_region2_bdd' 'cp_region2_bdd' 'mir1'});
    
    copy = workplane.geom.create('copy1', 'Copy');
    copy.selection('input').set({'uni1'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    copy = workplane.geom.create('copy2', 'Copy');
    copy.selection('input').set({'uni1'});
    copy.set('displx', num2str(-geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    mech = workplane.geom.create('pol1', 'Polygon');
    mech.set('source', 'table');
    for i=1:length(hex_x(1:end-1))
        mech.setIndex('table', num2str(hex2_x(i),'%10.9e'), i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(hex2_y(i),'%10.9e'), i-1, 1);
    end
    
    workplane.geom.create('dif1', 'Difference');
    workplane.geom.feature('dif1').selection('input').set({'pol1'});
    workplane.geom.feature('dif1').selection('input2').set({'uni1' 'copy1' 'copy2'});
    
    ext = geometry.feature.create('ext1', 'Extrude');
    ext.set('workplane', 'wp1');
    ext.selection('input').set({'wp1'});
    ext.setIndex('distance', 'd/2', 0);
      
    geometry.run;
    figure(1)
    mphgeom(model);
    view(0,90);
    mphsave(model,Eval_mech.filename)

%%   COMSOL-MATERIAL
    disp('Setting Material...');
    
    Silicon = model.material.create('mat2', 'Common', 'comp1');

    model.material('mat2').label('Si - Silicon (single-crystal, anisotropic) 1');
    model.material('mat2').set('family', 'custom');
    model.material('mat2').set('lighting', 'cooktorrance');
    model.material('mat2').set('specular', 'custom');
    model.material('mat2').set('customspecular', [0.7843137254901961 1 1]);
    model.material('mat2').set('fresnel', 0.9);
    model.material('mat2').set('roughness', 0.1);
    model.material('mat2').set('shininess', 200);
    model.material('mat2').set('diffuse', 'custom');
    model.material('mat2').set('customdiffuse', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('ambient', 'custom');
    model.material('mat2').set('customambient', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('fresnel', 0.7);
    model.material('mat2').set('roughness', 0.5);
    model.material('mat2').propertyGroup('def').set('density', '2330[kg/m^3]');
    model.material('mat2').propertyGroup.create('Anisotropic', 'Anisotropic');    
    model.material('mat2').propertyGroup('Anisotropic').set('D', {'166[GPa]' '64[GPa]' '166[GPa]' '64[GPa]' '64[GPa]' '166[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]'  ...
    '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]'  ...
    '80[GPa]'});
    
    epsilon = 3e-9;
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    si_domain = mphselectbox(model,'geom1',coordBox,'domain');
    Silicon.selection.set(si_domain);
   
    mphsave(model, Eval_mech.filename);
    
%%   COMSOL-MESH AND COMSOL-PHYSICS
    disp('Meshing and Setting Physics...');
    
    mesh = model.mesh.create('mesh1', 'geom1');
    physics = model.physics.create('solid', 'SolidMechanics', 'geom1');
    physics.feature('lemm1').set('SolidModel', 'Anisotropic');
    
    coordBox = [-1, 1; -1, 1; -epsilon, epsilon];
    sbz = mphselectbox(model,'geom1',coordBox,'boundary');
    if (Eval_mech.symm_z == +1)
        physics.feature.create('sym1', 'SymmetrySolid', 2);
        physics.feature('sym1').selection.set([sbz]);
    elseif (symm_z == -1)
        physics.feature.create('as1', 'Antisymmetry', 2);
        physics.feature('as1').selection.set([sbz]);
    end
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    l1 = mphselectbox(model,'geom1',coordBox,'boundary');
    l1_all_edge = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, epsilon];
    l1_edge_bot = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; geom.mech.d/2-epsilon, geom.mech.d/2+epsilon];
    l1_edge_top = mphselectbox(model,'geom1',coordBox,'edge');
    
    l1_edge = setdiff(l1_all_edge, horzcat(l1_edge_bot, l1_edge_top));
    
    coordBox = [geom.mech.amech - epsilon, geom.mech.amech + epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    l2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    freequad = mesh.create('fq1', 'FreeQuad');
    freequad.selection.set([l1]);
    freequad.create('dis1', 'Distribution');
    freequad.feature('dis1').selection.set([l1_edge]);
    freequad.feature('dis1').set('numelem', 2);
    copy_face = mesh.create('cpf1', 'CopyFace');
    copy_face.selection('source').set([l1]);
    copy_face.selection('destination').set([l2]);
    
    physics.feature.create('pc1', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc1').set('PeriodicType', 'Floquet');
    physics.feature('pc1').selection.set([l1 l2]);
    
    geometry.feature.create('rot1', 'Rotate');
    geometry.feature('rot1').set('rot', '60');
    geometry.feature('rot1').set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    geometry.feature('rot1').selection('input').set({'ext1'});
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    m1 = mphselectbox(model,'geom1',coordBox,'boundary');
    m1_all_edge = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, epsilon];
    m1_edge_bot = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; geom.mech.d/2-epsilon, geom.mech.d/2+epsilon];
    m1_edge_top = mphselectbox(model,'geom1',coordBox,'edge');
    
    m1_edge = setdiff(m1_all_edge, horzcat(m1_edge_bot, m1_edge_top));
    
    coordBox = [geom.mech.amech - epsilon, geom.mech.amech + epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    m2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    freequad = mesh.create('fq2', 'FreeQuad');
    freequad.selection.set([m1]);
    freequad.create('dis1', 'Distribution');
    freequad.feature('dis1').selection.set([m1_edge]);
    freequad.feature('dis1').set('numelem', 2);    
    copy_face = mesh.create('cpf2', 'CopyFace');
    copy_face.selection('source').set([m1]);
    copy_face.selection('destination').set([m2]);
    
    physics.feature.create('pc2', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc2').set('PeriodicType', 'Floquet');
    physics.feature('pc2').selection.set([m1 m2]);
    
    geometry.feature.create('rot2', 'Rotate');
    geometry.feature('rot2').set('rot', '60');
    geometry.feature('rot2').set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    geometry.feature('rot2').selection('input').set({'rot1'});
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    n1 = mphselectbox(model,'geom1',coordBox,'boundary');
    n1_all_edge = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, epsilon];
    n1_edge_bot = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; geom.mech.d/2-epsilon, geom.mech.d/2+epsilon];
    n1_edge_top = mphselectbox(model,'geom1',coordBox,'edge');
    
    n1_edge = setdiff(n1_all_edge, horzcat(n1_edge_bot, n1_edge_top));
    
    coordBox = [geom.mech.amech - epsilon, geom.mech.amech + epsilon; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    n2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    freequad = mesh.create('fq3', 'FreeQuad');
    freequad.selection.set([n1]);
    freequad.create('dis1', 'Distribution');
    freequad.feature('dis1').selection.set([n1_edge]);
    freequad.feature('dis1').set('numelem', 2);
    copy_face = mesh.create('cpf3', 'CopyFace');
    copy_face.selection('source').set([n1]);
    copy_face.selection('destination').set([n2]);
    
    physics.feature.create('pc3', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc3').set('PeriodicType', 'Floquet');
    physics.feature('pc3').selection.set([n1 n2]);
    
    geometry.feature.create('rot3', 'Rotate');
    geometry.feature('rot3').set('rot', '-120');
    geometry.feature('rot3').set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    geometry.feature('rot3').selection('input').set({'rot2'});
    
    mesh.create('ftri1', 'FreeTri');
    mesh.feature('ftri1').create('size1', 'Size');
    mesh.feature('ftri1').feature('size1').set('custom', 'on');
    mesh.feature('ftri1').feature('size1').set('hmax', 'hmax');
    mesh.feature('ftri1').feature('size1').set('hmin', 'hmin');
    mesh.feature('ftri1').selection.set([sbz]);
    
    mesh.create('swe1', 'Sweep');
    mesh.feature('swe1').create('dis1', 'Distribution');
    mesh.feature('swe1').feature('dis1').set('numelem', num2str(2)); %Number of mesh layers = 2
    mesh.run;
    figure(2)
    mphmesh(model)
    mphsave(model, Eval_mech.filename);

%%   COMSOL STUDY

    disp('Launching study... ');
    study = model.study.create('std');
    studyEf = study.feature.create('eig', 'Eigenfrequency');
    
    studyEf.activate('solid', true);
    studyEf.set('neigs', num2str(Eval_mech.NumberOfFrequencies));
    studyEf.set('shift', num2str(Eval_mech.CentralFrequency));
    
    for i=1:size(Eval_mech.kx,2)
%     for i=1:25
        disp( ['Iteration: ',num2str(i),'/',num2str(size(Eval_mech.kx,2))]  )
        physics.feature('pc1').set('kFloquet', {num2str(Eval_mech.kx(i),'%10.9e') num2str(Eval_mech.ky(i),'%10.9e') '0'});
        physics.feature('pc2').set('kFloquet', {num2str(Eval_mech.kx(i),'%10.9e') num2str(Eval_mech.ky(i),'%10.9e') '0'});
        physics.feature('pc3').set('kFloquet', {num2str(Eval_mech.kx(i),'%10.9e') num2str(Eval_mech.ky(i),'%10.9e') '0'});
        study.run;
        data = mpheval(model,{'freq'});
        Eval_mech.bands(:,i)=data.d1(:,1);
        
        save(Eval_mech.filename, 'Eval_mech', 'geom');
        
%         if i==2*Eval_mech.res-1
%            mphsave(model,strcat(Eval_mech.filename,'_K_point')); 
%         end
    end
    
%     mphsave(model, Eval_mech.filename);
    Eval_mech.time = toc;
    
    save(Eval_mech.filename, 'Eval_mech', 'geom');
%%     
end